<?php

namespace Tests\Feature;

use App\Models\Order;
use App\Models\Outlet;
use App\Models\Product;
use App\Models\Register;
use App\Models\User;
use Illuminate\Foundation\Testing\RefreshDatabase;
use Illuminate\Support\Str;
use Tests\TestCase;

class OperationsFlowTest extends TestCase
{
    use RefreshDatabase;

    public function test_split_payment_is_saved(): void
    {
        $outlet = Outlet::query()->create(['name' => 'Outlet A', 'timezone' => 'Asia/Jakarta']);
        $register = Register::query()->create(['outlet_id' => $outlet->id, 'name' => 'Kasir 1']);
        $cashier = User::factory()->create(['outlet_id' => $outlet->id, 'role' => 'cashier']);
        $product = Product::query()->create([
            'outlet_id' => $outlet->id,
            'name' => 'Mie Goreng',
            'sku' => 'MIE-1',
            'base_price' => 20000,
            'is_active' => true,
        ]);

        $this->actingAs($cashier)->postJson('/pos/orders', [
            'client_order_id' => (string) Str::uuid(),
            'register_id' => $register->id,
            'order_type' => 'dine-in',
            'items' => [
                ['product_id' => $product->id, 'qty' => 1, 'addon_ids' => []],
            ],
            'payments' => [
                ['method' => 'CASH', 'amount' => 10000],
                ['method' => 'QRIS', 'amount' => 13200],
            ],
        ])->assertOk();

        $this->assertDatabaseCount('payments', 2);
        $this->assertDatabaseHas('payments', ['kind' => 'PAYMENT', 'method' => 'CASH']);
        $this->assertDatabaseHas('payments', ['kind' => 'PAYMENT', 'method' => 'QRIS']);
    }

    public function test_owner_can_refund_and_update_kitchen_status(): void
    {
        $outlet = Outlet::query()->create(['name' => 'Outlet A', 'timezone' => 'Asia/Jakarta']);
        $owner = User::factory()->create(['outlet_id' => $outlet->id, 'role' => 'owner']);
        $order = Order::query()->create([
            'client_order_id' => (string) Str::uuid(),
            'outlet_id' => $outlet->id,
            'register_id' => Register::query()->create(['outlet_id' => $outlet->id, 'name' => 'Kasir 1'])->id,
            'user_id' => $owner->id,
            'status' => 'PAID',
            'kitchen_status' => 'NEW',
            'order_type' => 'dine-in',
            'subtotal' => 10000,
            'discount' => 0,
            'tax' => 0,
            'service' => 0,
            'total' => 10000,
            'receipt_number' => 'INV-20260222-0001',
            'paid_at' => now(),
        ]);

        $this->actingAs($owner)->post("/kds/{$order->id}/status", [
            'kitchen_status' => 'READY',
        ])->assertRedirect();

        $this->assertDatabaseHas('orders', ['id' => $order->id, 'kitchen_status' => 'READY']);

        $this->actingAs($owner)->post("/orders/{$order->id}/refund", [
            'amount' => 5000,
            'reason' => 'Customer complaint',
            'method' => 'CASH',
        ])->assertRedirect();

        $this->assertDatabaseHas('orders', ['id' => $order->id, 'refund_total' => 5000]);
        $this->assertDatabaseHas('payments', ['order_id' => $order->id, 'kind' => 'REFUND', 'amount' => 5000]);
    }
}
