<?php

namespace App\Http\Controllers;

use App\Http\Requests\VariantGroupRequest;
use App\Http\Requests\VariantOptionRequest;
use App\Models\Product;
use App\Models\ProductVariant;
use App\Models\VariantOption;
use Illuminate\Http\RedirectResponse;
use Illuminate\Support\Facades\Cache;

class ProductVariantController extends Controller
{
    public function store(VariantGroupRequest $request, Product $product): RedirectResponse
    {
        abort_if((int) $product->outlet_id !== (int) $request->user()->outlet_id, 403);
        $product->variants()->create([
            'name' => $request->validated('name'),
            'is_required' => $request->boolean('is_required'),
            'max_selection' => (int) ($request->validated('max_selection') ?? 1),
            'sort_order' => (int) ($request->validated('sort_order') ?? 0),
        ]);
        Cache::forget("pos.catalog.{$request->user()->outlet_id}");

        return back()->with('success', 'Variant group ditambahkan.');
    }

    public function update(VariantGroupRequest $request, ProductVariant $variant): RedirectResponse
    {
        abort_if((int) $variant->product->outlet_id !== (int) $request->user()->outlet_id, 403);
        $variant->update([
            'name' => $request->validated('name'),
            'is_required' => $request->boolean('is_required'),
            'max_selection' => (int) ($request->validated('max_selection') ?? 1),
            'sort_order' => (int) ($request->validated('sort_order') ?? 0),
        ]);
        Cache::forget("pos.catalog.{$request->user()->outlet_id}");

        return back()->with('success', 'Variant group diperbarui.');
    }

    public function destroy(ProductVariant $variant): RedirectResponse
    {
        abort_if((int) $variant->product->outlet_id !== (int) auth()->user()->outlet_id, 403);
        Cache::forget("pos.catalog.".auth()->user()->outlet_id);
        $variant->delete();

        return back()->with('success', 'Variant group dihapus.');
    }

    public function storeOption(VariantOptionRequest $request, ProductVariant $variant): RedirectResponse
    {
        abort_if((int) $variant->product->outlet_id !== (int) $request->user()->outlet_id, 403);
        $variant->options()->create($request->validated());
        Cache::forget("pos.catalog.{$request->user()->outlet_id}");

        return back()->with('success', 'Variant option ditambahkan.');
    }

    public function updateOption(VariantOptionRequest $request, VariantOption $option): RedirectResponse
    {
        abort_if((int) $option->variant->product->outlet_id !== (int) $request->user()->outlet_id, 403);
        $option->update($request->validated());
        Cache::forget("pos.catalog.{$request->user()->outlet_id}");

        return back()->with('success', 'Variant option diperbarui.');
    }

    public function destroyOption(VariantOption $option): RedirectResponse
    {
        abort_if((int) $option->variant->product->outlet_id !== (int) auth()->user()->outlet_id, 403);
        Cache::forget("pos.catalog.".auth()->user()->outlet_id);
        $option->delete();

        return back()->with('success', 'Variant option dihapus.');
    }
}
