<?php

namespace App\Http\Controllers;

use App\Http\Requests\ProductRequest;
use App\Models\Category;
use App\Models\Product;
use Illuminate\Http\RedirectResponse;
use Illuminate\Support\Facades\Cache;
use Illuminate\Support\Facades\Storage;
use Illuminate\View\View;

class ProductController extends Controller
{
    public function index(): View
    {
        $outletId = auth()->user()->outlet_id;
        $products = Product::query()
            ->with('category')
            ->where('outlet_id', $outletId)
            ->latest()
            ->paginate(20);
        $categories = Category::query()
            ->where('outlet_id', $outletId)
            ->orderBy('sort_order')
            ->get();

        return view('master.products.index', compact('products', 'categories'));
    }

    public function store(ProductRequest $request): RedirectResponse
    {
        $data = $request->safe()->except(['is_active', 'image_file']);
        $data['outlet_id'] = $request->user()->outlet_id;
        $data['is_active'] = $request->boolean('is_active', true);

        if ($request->hasFile('image_file')) {
            $path = $request->file('image_file')->store('products', 'public');
            $data['image_url'] = Storage::url($path);
        }

        Product::query()->create([
            ...$data,
        ]);
        Cache::forget("pos.catalog.{$request->user()->outlet_id}");

        return back()->with('success', 'Produk ditambahkan.');
    }

    public function edit(Product $product): View
    {
        abort_if((int) $product->outlet_id !== (int) auth()->user()->outlet_id, 403);
        $product->load(['variants.options', 'addons']);
        $categories = Category::query()
            ->where('outlet_id', auth()->user()->outlet_id)
            ->orderBy('sort_order')
            ->get();

        return view('master.products.edit', compact('product', 'categories'));
    }

    public function update(ProductRequest $request, Product $product): RedirectResponse
    {
        abort_if((int) $product->outlet_id !== (int) $request->user()->outlet_id, 403);
        $data = $request->safe()->except(['is_active', 'image_file']);
        $data['is_active'] = $request->boolean('is_active');

        if ($request->hasFile('image_file')) {
            $path = $request->file('image_file')->store('products', 'public');
            $data['image_url'] = Storage::url($path);
        }

        $product->update([
            ...$data,
        ]);
        Cache::forget("pos.catalog.{$request->user()->outlet_id}");

        return back()->with('success', 'Produk diperbarui.');
    }

    public function destroy(Product $product): RedirectResponse
    {
        abort_if((int) $product->outlet_id !== (int) auth()->user()->outlet_id, 403);
        Cache::forget("pos.catalog.".auth()->user()->outlet_id);
        $product->delete();

        return back()->with('success', 'Produk dihapus.');
    }
}
